/*	This file contains miscellaneous definitions to make the FFT
	demonstration work.  Primarily, "complex" and "ComplexArray" types
	are defined to make using complex numbers flexible.
*/


/*	WARNING:  The definitions in this file are NOT intended for use in
	any application.  They expose class members unduly and are generally
	unsafe.  This file exists soley for the purpose of making the FFT
	demonstration code execute.  That illustrates the design so that
	real code can be written.
*/


/*	Although C (as of C9X) and C++ have complex types, here is a simple
	implementation that provides all we need.
*/
class complex {
public:
	// Expose the real and imaginary components.
	float re, im;
	
	// Make a complex from a real or a real and an imaginary.
	complex(float re, float im = 0.) : re(re), im(im) {}
	
	// Add and multiply complex numbers.
	complex operator +(complex b) const
	{
		return complex(re + b.re, im + b.im);
	}
	complex operator *(complex b) const
	{
		return complex(re*b.re - im*b.im, re*b.im + im*b.re);
	}
	
	// Add to a complex number in place.
	complex operator +=(complex b)
	{
		re += b.re;
		im += b.im;
		return *this;
	}
};


/*	ComplexArray provides a flexible interface to an array of complex
	numbers.  It allows us to extract either all the real or imaginary
	components (through members .re and .im) or to extract a single
	complex member (through the subscript operator).
	
	This permits the demonstration code to ignore the implementation;
	it does not need to know if complex numbers are stored with
	separated or interleaved data.  When implementing a high-performance
	FFT, a choice will be made, and particular code for it will be written.
*/
class ComplexArray {
private:
	/*	ComplexProxy provides ways to treat references to elements of a
		ComplexArray as if they were complex numbers.
	*/
	class ComplexProxy {
	public:
		/*	The real and imaginary components are references to the actual
			objects in the arrays in the ComplexArray.
		*/
		float &re, &im;
		
		// A proxy is constructed from a ComplexArray and a subscript.
		ComplexProxy(ComplexArray &array, int i)
			: re(array.re[i]), im(array.im[i]) {}
		
		// Convert a proxy to a complex number just by copying.
		operator complex() { return complex(re, im); }
		
		// Allow assignment to the proxy.
		ComplexProxy &operator =(complex b)
		{
			re = b.re; im = b.im;
			return *this;
		}
		ComplexProxy &operator =(ComplexProxy b)
		{
			re = b.re; im = b.im;
			return *this;
		}
	};

public:
	// Expose the arrays of real and imaginary components.
	float	*re, *im;
	
	// Make a complex array from two real arrays.
	ComplexArray(float *re, float *im) : re(re), im(im) {}
	
	// Return an individual complex element.
	ComplexProxy operator [](int i) { return ComplexProxy(*this, i); }
		
	// Allow arithmetic on the array address.
	ComplexArray operator +(int n) const
	{
		return ComplexArray(re + n, im + n);
	}
};


/*	This convenience routine frees a bunch of things we have allocated:
		I elements in *v,
		*v itself,
		*m, and
		*n.
*/
static void FreeVectors(ComplexArray **v, int **m, int **n, int I)
{
	int i;

	for (i = 0; i < I; ++i) {
		free((*v)[i].im);
		free((*v)[i].re);
	}

	free(*v);
	free(*m);
	free(*n);
}


/*	AllocVectors allocates P elements for *m, P+1 elements for *n, and
	N+1 for *v.  Then, for each element in *v, it allocates space for the
	re and im members in the element.
*/
static int AllocVectors(ComplexArray **v, int **m, int **n, int N, int P)
{
	int i;

	*m = (int *) malloc(P * sizeof **m);
	*n = (int *) malloc((P+1) * sizeof **n);
	*v = (ComplexArray *) malloc((N+1) * sizeof **v);
	if (*v == NULL || *m == NULL || *n == NULL) {
		FreeVectors(v, m, n, 0);
		fprintf(stderr, "Error, failed to allocate memory.\n");
		return 1;
	}

	for (i = 0; i <= N; ++i) {
		/*	Yuck, malloc for C++ objects.  However, we are keeping the code
			simple, quick, and dirty.
		*/
		(*v)[i] = ComplexArray(	(float *) malloc((1<<N) * sizeof (*v)[i].re),
								(float *) malloc((1<<N) * sizeof (*v)[i].im));
		if ((*v)[i].re == NULL || (*v)[i].im == NULL) {
			FreeVectors(v, m, n, i+1);
			fprintf(stderr, "Error, failed to allocate memory.\n");
			return 1;
		}
	}

	return 0;
}
