/*	This module contains definitions common to all of the FFT demonstration
	code, such as TwoPi and simple subroutines like log2 and one.
	
	It also includes "glue.h", the file that provides some utility
	definitions to make the demonstration code work and look simple.
*/

#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"


// Two times pi.
static const double TwoPi = 2 * 3.1415926535897932384626433;


// Bit-reverse a full word (32 bits).
static unsigned int rw(unsigned int k)
{
	// This is a table of bit-reversals of bytes.
	static const unsigned char b[256] = {
  0, 128,  64, 192,  32, 160,  96, 224,  16, 144,  80, 208,  48, 176, 112, 240,
  8, 136,  72, 200,  40, 168, 104, 232,  24, 152,  88, 216,  56, 184, 120, 248,
  4, 132,  68, 196,  36, 164, 100, 228,  20, 148,  84, 212,  52, 180, 116, 244,
 12, 140,  76, 204,  44, 172, 108, 236,  28, 156,  92, 220,  60, 188, 124, 252,
  2, 130,  66, 194,  34, 162,  98, 226,  18, 146,  82, 210,  50, 178, 114, 242,
 10, 138,  74, 202,  42, 170, 106, 234,  26, 154,  90, 218,  58, 186, 122, 250,
  6, 134,  70, 198,  38, 166, 102, 230,  22, 150,  86, 214,  54, 182, 118, 246,
 14, 142,  78, 206,  46, 174, 110, 238,  30, 158,  94, 222,  62, 190, 126, 254,
  1, 129,  65, 193,  33, 161,  97, 225,  17, 145,  81, 209,  49, 177, 113, 241,
  9, 137,  73, 201,  41, 169, 105, 233,  25, 153,  89, 217,  57, 185, 121, 249,
  5, 133,  69, 197,  37, 165, 101, 229,  21, 149,  85, 213,  53, 181, 117, 245,
 13, 141,  77, 205,  45, 173, 109, 237,  29, 157,  93, 221,  61, 189, 125, 253,
  3, 131,  67, 195,  35, 163,  99, 227,  19, 147,  83, 211,  51, 179, 115, 243,
 11, 139,  75, 203,  43, 171, 107, 235,  27, 155,  91, 219,  59, 187, 123, 251,
  7, 135,  71, 199,  39, 167, 103, 231,  23, 151,  87, 215,  55, 183, 119, 247,
 15, 143,  79, 207,  47, 175, 111, 239,  31, 159,  95, 223,  63, 191, 127, 255
	};
	unsigned char
		b0 = b[k >> 0*8 & 0xff],
		b1 = b[k >> 1*8 & 0xff],
		b2 = b[k >> 2*8 & 0xff],
		b3 = b[k >> 3*8 & 0xff];
	return b0 << 3*8 | b1 << 2*8 | b2 << 1*8 | b3 << 0*8;
}


// Calculate "r" function, as defined in the FFT construction paper.
static float r(unsigned int k)
{
 	return 1./4294967296. * rw(k);
}


// Base-two logarithm of x.
static int ilog2(unsigned int n)
{
 	int c;
 	for (c = 0; n >>= 1; ++c)
 		;
   return c;
}


// "One to the power of x" is e to the power of 2*pi*i*x.
static complex one(float x)
{
	return complex(cos(TwoPi*x), sin(TwoPi*x));
}


// This is a utility routine for FFT_Butterflies, below.
static ComplexArray CopyOfPortionOfComplexArray(ComplexArray vIn,
	int k0, int c0)
{
	float *re = (float *) malloc(c0 * sizeof *re);
	float *im = (float *) malloc(c0 * sizeof *im);
	ComplexArray result(re, im);
	int k;
	
	if (re == NULL || im == NULL)
	{
		fputs("Fatal error, unable to allocate memory for array copy.\n",
			stderr);
		free(im);
		free(re);
		exit(1);
	}

	for (k = 0; k < c0; ++k)
		result[k] = vIn[c0*k0 + k];

	return result;
}


// This is a utility routine for FFT_Butterflies, below.
static void FreeCopy(ComplexArray copy)
{
	free(copy.im);
	free(copy.re);
}


/*	FFT_Butterflies is included in this file to make it easier to build
	a sequence of demonstrations.  It acts as a filler that provides
	the mathematical calculations before efficient code is built.  It
	should NOT be used in any final design.
*/
// FFT_Butterflies
static void FFT_Butterflies(
	int m,				// Butterfly radix.
	ComplexArray vOut,	// Address of output vector.
	ComplexArray vIn,	// Address of input vector.
	int k0,				// k0 from equation.
	int c0				// Coefficient for k0.
)
{
	// Coefficient for k1 is coefficient for k0 divided by 1<<m.
	const int c1 = c0 >> m;
 	int j1, k1, k2;

	/*	This is not shown in the paper, but we will make this routine
		work in place by copying all the inputs to an auxiliary array.
		The expression In[c1*j1 + k2] replaces vIn[c0*k0 + c1*j1 + k2].
		Mathematically, the routine is identical to that in the paper.
	*/
	ComplexArray In = CopyOfPortionOfComplexArray(vIn, k0, c0);

	for (k2 = 0; k2 < c1  ; ++k2)
	for (k1 = 0; k1 < 1<<m; ++k1)
	{
		complex sum = 0.;
		for (j1 = 0; j1 < 1<<m; ++j1)
			sum += one(j1*r(k1)) * one(j1*r((1<<m)*k0)) * In[c1*j1+k2];
		vOut[c0*k0 + c1*k1 + k2] = sum;
	}

	FreeCopy(In);
}


// FinalIndices
typedef struct {
	unsigned short int read, write;
} FinalIndices;


// Auxiliary routine to construct FinalIndices.  Unneeded in C 1999.
static FinalIndices Construct(unsigned int read, unsigned int write)
{
	FinalIndices result = { read, write };
	return result;
}
